<?php

namespace App\Http\Controllers\api\v1;

use Exception;
use App\Models\User;
use App\Models\api\v1\Role;
use Illuminate\Support\Str;
use App\Models\api\v1\Order;
use App\Models\api\v1\Space;
use App\Models\api\v1\Stock;
use Illuminate\Http\Request;
use App\Models\api\v1\Device;
use App\Services\CurlService;
use App\Models\api\v1\LineItem;
use App\Models\api\v1\UserRole;
use App\Enums\OrderStatusesEnum;
use App\Services\PaymentService;
use Illuminate\Http\JsonResponse;
use App\Models\api\v1\OrderStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Helpers\NotificationHelpers;
use App\Http\Controllers\Controller;
use App\Models\api\v1\Item;
use App\Models\api\v1\UsersRolesSpace;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{

    public function userOrders(Request $request): JsonResponse
    {
        try {
            return $this->ReturnResponse(
                true,
                "User Order list",
                Order::select('orders.*')
                    ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                    ->OrderBy('orders.created_at', 'asc')
                    ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                    ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                    ->with(['orderStatus', 'lineItems.stock.item'])
                    ->distinct()
                    ->get()
                    ->toArray()
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders for User" . $ex->getMessage();
            Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }

    public function userOrdersWithSpace(Request $request): JsonResponse
    {
        try {
            return $this->ReturnResponse(
                true,
                "Space Order list",
                [
                    "space" => Order::select('orders.*')
                        ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                        ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                        ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                        ->with(['orderStatus', 'lineItems.stock.item'])
                        ->OrderBy('orders.created_at', 'desc')
                        ->first()->lineItems?->first()?->stock?->space ?? "",
                    "user_orders" =>
                    Order::select('orders.*')
                        ->where([['orders.created_by', request()->user()->id], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                        ->OrderBy('orders.created_at', 'desc')
                        ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                        ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                        ->with(['orderStatus', 'lineItems.stock.item'])
                        ->get()
                        ->toArray()
                ]
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders for User" . $ex->getMessage();
            Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    public function spaceOrders(Request $request, string $space_uuid): JsonResponse
    {

        try {
            $space = Space::where([['uuid', $space_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
            if (!$space) {
                return $this->ReturnResponse(false, "Space not found", [], 404);
            }

            return $this->ReturnResponse(
                true,
                "Space Order list",
                Order::getOrdersForSpace($space->id)
            );
        } catch (Exception $ex) {
            $message =  "Unable to Retrieve list of orders" . $ex->getMessage();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    /**
     * Cette fonction permet d'avoir la liste des status des commandes
     * Vous pouvez avoir les status suivant l'état que vous voulez voir
     * 
     * @return mixed
     */
    public function getstatuses(Request $request): JsonResponse
    {
        $statuses           = OrderStatus::orderBy('step_number')->get()->toArray();
        $step = $request->get('after_step');
        if (isset($step)) {

            $statuses = OrderStatus::where('step_number', intval($step) + 1)->orderBy('step_number')->get()->toArray();
        }
        return $this->ReturnResponse(true, "Liste des status de commandes", $statuses);
    }

    /**
     * Cette fonction permet de mettre a jour le status d'une commande
     * 
     * @return mixed
     */
    public function changeStatus(Request $request, string $order_uuid, string $status): JsonResponse
    {

        try {
            DB::beginTransaction();

            $order = Order::select('orders.*')->Where([['orders.uuid', $order_uuid], ['orders.is_deleted', 0], ['orders.is_active', 1]])
                ->join('line_items', 'line_items.order_id', '=', 'orders.id')
                ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
                ->first();

            if (!$order) {
                return $this->ReturnResponse(false, "Order not found", [], 404);
            }

            $currentStatus      = OrderStatus::where('id', $order->order_status_id)->first();
            $assignStatus       = OrderStatus::where('name', strtoupper($status))->first();
            if (!$assignStatus) {
                return $this->ReturnResponse(false, "Le status de commande inconnu", [], 404);
            }

            // @TODO: Add check for status back ASAP
            //        if ($assignStatus->step_number == $currentStatus->step_number || $assignStatus->step_number == $currentStatus->step_number + 1)
            //        {

            $userInSpace = UsersRolesSpace::where('space_id', $order->space_id)->get()->pluck('user_role_id ');
            $userRoles = UserRole::whereIn('id', $userInSpace)->where('role_id', 3)->first();
            $message = '';
            if ($assignStatus->name == "SERVED"  || $assignStatus->name == "PAID") {
                $validator = Validator::make(
                    $request->all(),
                    [
                        'payment_method' => 'required',
                    ]
                );

                // If Errors
                if ($validator->fails()) {
                    return $this->ReturnResponse(false, $validator->errors(), [], 401);
                }
                //@TODO : en attente du mode de paiement

                if ($order->remaining_amount < 0) {

                    if (request()->header('mobileId')) {
                        NotificationHelpers::send(
                            request()->header('mobileId'),
                            "Information sur la commande ",
                            "Merci de vérifier le montant de la commande",
                        );
                    }

                    return $this->ReturnResponse(
                        false,
                        "Merci de vérifier le montant de la commande",
                        [],
                        408
                    );
                }
                if (strtoupper($request->payment_method) == 'FLEX') {

                    $access_token = $request->header('Authorization');
                    $auth_header = explode(' ', $access_token);
                    $token = $auth_header[1];

                    $data = [

                        "montant"       => $order->remaining_amount,
                        "merchant"      => $userRoles?->user?->id  ?? 536,
                        "commentaire"   => "Test de paiement marchand"

                    ];
                    $payment = new PaymentService(env('FLEX_API_URL_MERCHANT'), $token);
                    $return = json_decode($payment->sendRequest('POST', $data));

                    if ($return->success == false) {

                        if (request()->header('mobileId')) {
                            NotificationHelpers::send(
                                request()->header('mobileId'),
                                "Information sur la commande ",
                                "Paiement Flex a échoué : " . $return->message
                            );
                        }
                        if (Device::where('user_id', $userRoles?->user?->id)->first()) {
                            NotificationHelpers::send(
                                Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
                                "Information sur la commande ",
                                "Paiement Flex a échoué : " . $return->message
                            );
                        }

                        return $this->ReturnResponse(
                            false,
                            $return->message,
                            [],
                            408
                        );
                    }

                    $message = ' .' . $return->message;
                }
                $order->order_status_id = $assignStatus->id;
                $order->updated_at      = now();
                $order->updated_by      = $request->user()->id;
                $order->payment_method    = $request->payment_method;

                $order->save();
                DB::commit();

                if (request()->header('mobileId')) {
                    NotificationHelpers::send(
                        request()->header('mobileId'),
                        "Information sur la commande ",
                        "Votre paiement a été pris en compte" . $message
                    );
                }
                if (Device::where('user_id', $userRoles?->user?->id)->first()) {
                    NotificationHelpers::send(
                        Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
                        "Information sur la commande ",
                        "Un paiement a été pris en compte" . $message
                    );
                }
                return $this->ReturnResponse(
                    true,
                    "Votre paiement a été pris en compte" . $message,
                    [$order]
                );
            }
            $order->order_status_id = $assignStatus->id;
            $order->updated_at      = now();
            $order->updated_by      = $request->user()->id;
            $order->save();

            if (request()->header('mobileId')) {
                NotificationHelpers::send(
                    request()->header('mobileId'),
                    "Information sur la commande ",
                    "Le statut de la commande est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                );
            }
            if (Device::where('user_id', $userRoles?->user?->id)->first()) {
                NotificationHelpers::send(
                    Device::where('user_id', $userRoles?->user?->id)->first()->device_token,
                    "Information sur la commande ",
                    "Le statut de la commande est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                );
            }

            DB::commit();
            return $this->ReturnResponse(
                true,
                "Le statut de la commande est passé de " . $currentStatus->name . " à " . $assignStatus->name,
                [$order]
            );

            //        }git 


        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossible de modifier cette commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }



    public function show(Request $request, string $uuid): JsonResponse
    {
        $order = Order::Where([['uuid', $uuid], ['is_deleted', 0], ['is_active', 1]])
            ->with(['orderStatus', 'lineItems.stock.item'])->first();
        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        return $this->ReturnResponse(
            true,
            "Order Details",
            [$order]
        );
    }

    public function show_space(Request $request, string $space_uuid, string $uuid): JsonResponse
    {
        return $this->show($request, $uuid);
    }

    public function create(Request $request): JsonResponse
    {

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
                "table_number"             => "string",
            ]
        );

        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }

        $order = Order::where([
            ['created_by', $request->user()->id],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->with(['orderStatus', 'lineItems'])->first();


        $amount_due = 0;
        if ($order) {
            $order->order_status_id =  OrderStatus::where('name', 'CANCELED')->first()->id;
            $order->save();
        }
        $order = Order::create([
            "order_status_id"       => 1,
            "uuid"                  => Str::uuid(),
            "amount_due"            => $amount_due,
            "table_number"          => $request->table_number,
            "remaining_amount"      => $amount_due,
            "created_by"            => $request->user()->id,
            "updated_by"            => $request->user()->id,
            "payment_method"           => $request->payment_method,
            "is_active"             => 1
        ]);

        // if (!$order) {
        //     $order = Order::create([
        //         "order_status_id"       => 1,
        //         "uuid"                  => Str::uuid(),
        //         "amount_due"            => $amount_due,
        //         "table_number"          => $request->table_number,
        //         "remaining_amount"      => $amount_due,
        //         "created_by"            => $request->user()->id,
        //         "updated_by"            => $request->user()->id,
        //         "payment_method"           => $request->payment_method,
        //         "is_active"             => 1
        //     ]);
        // } else {
        //     return $this->ReturnResponse(false, "Il existe déjà un panier. Merci de mettre à jour le panier", [], 400);
        // }


        $stock_id_list   = [];
        $list_line_items = [];
        foreach ($request->line_items as $key => $sline) {
            $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"];
        }

        try {
            DB::beginTransaction();
            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();


            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }
            $amount_due = 0;

            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];
                $amount_due += $stock->price_after_sale *  $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }


            LineItem::insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            //$order->lineItems()->insert($list_line_items); // inserrton par lot de 10 dans la boucle 
            $order->update([
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due,
                "payment_method"    => $request->payment_method,

            ]);
            /*"*/
            DB::commit();

            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();


            return $this->ReturnResponse(true, "La commande a été enregistrée", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossible de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }

    public function assignOrderToWaiter(Request $request, string $space_uuid, string $order_uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $order_uuid],
            ['is_deleted', 0],
            ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        try {
            DB::beginTransaction();
            $order->server_id   = $request->user()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();


            if (request()->header('mobileId')) {
                NotificationHelpers::sendNotification(
                    request()->header('mobileId'),
                    "Information sur la commande ",
                    "Votre commande est en cours de traitement"
                );
            }

            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }
    }


    public function assignOrderByWaiter(Request $request, string $space_uuid, string $order_uuid, string $waiter_uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $order_uuid],
            ['is_deleted', 0],
            ['order_status_id', '!=', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        try {
            DB::beginTransaction();
            $order->server_id   = User::where('uuid', $waiter_uuid)->first()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();
            DB::commit();
            return $this->ReturnResponse(true, "Order assigned", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }
    }


    public function OrderToWaiter(Request $request, string $space_uuid, string $waiter_uuid): JsonResponse
    {
        $waiter = User::where([
            ['uuid', $waiter_uuid],
            ['blocked', 0],
            ['is_active', 1],
            ['is_deleted', 0]
        ])->first();

        if (!$waiter) {
            return $this->ReturnResponse(false, "Waiter not found", [], 404);
        }
        $waiter_role = UserRole::where([
            ['is_active', 1],
            ['is_deleted', 0],
            ['user_id', $waiter->id],
            ['role_id', Role::where('name', 'Serveur')->first()->id]
        ])->first();
        if (!$waiter_role) {
            return $this->ReturnResponse(false, "USer was not waiterw", [], 404);
        }

        $space = Space::where('uuid', $space_uuid)->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }

        $waiter_in_space = UsersRolesSpace::where([
            ['space_id', $space->id],
            ['user_role_id', $waiter_role->id]
        ])->first();
        if (!$waiter_in_space) {
            return $this->ReturnResponse(false, "user not waiter in this space", [], 404);
        }

        return $this->ReturnResponse(true, "Order to waiter", [
            "Assignées" => Order::where('server_id', $waiter->id)->get(),
            "Attente" => Order::where([['server_id', $waiter->id], ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]])->get(),
            "Livrées" => Order::where([['server_id', $waiter->id], ['order_status_id', OrderStatus::where('name', 'SERVED')->first()->id]])->get(),
        ], 200);
    }


    public function OrderToSpace(Request $request, string $space_uuid): JsonResponse
    {
        $space = Space::where('uuid', $space_uuid)->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }

        return $this->ReturnResponse(true, "Order list by order status", [
            'Attente' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::where('name', 'CONFIRMED')->first()->id),
            'Livrée' => Order::getOrdersForSpaceByStatus($space->id, OrderStatus::where('name', 'SERVED')->first()->id)
        ], 200);
    }

    public function update(Request $request, $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();
        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'table_number'             => 'string',
                'rebate'                   => 'numeric|min:0',
            ]
        );

        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }

        try {
            DB::beginTransaction();
            $order->update([
                'table_number'          => $request->table_number
            ]);
            if (!is_null($request->rebate)) {
                $order->update(
                    [
                        'rebate'                => $request->rebate,
                        'remaining_amount'      => $order->amount_due - intval($request->rebate)
                    ]
                );
            }
            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", []);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }

    public function updateSpace(Request $request, string $space_uuid, string $uuid): JsonResponse
    {
        return $this->update($request, $uuid);
    }


    public function updateOrderLineItem(Request $request, $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
            ]
        );
        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }


        $stock_id_list   = [];
        $list_line_items = [];

        foreach ($request->line_items as $key => $sline) {
            $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"]; // si deux element identique faire la somme
        }

        try {
            DB::beginTransaction();

            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();

            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }

            $order->lineItems()->delete();

            $amount_due = 0;


            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];

                $amount_due += $stock->price_after_sale * $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }

            LineItem::insert($list_line_items);

            $order->update([
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due - $order->rebate,
            ]);
            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();
            DB::commit();
            return $this->ReturnResponse(true, "La commande a été mise à jour", [$order->with('lineItems')]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }


    public function updateOrderLineItemSpace(Request $request,  string $space_uuid, string $uuid): JsonResponse
    {
        //return $this->updateOrderLineItem($request, $uuid);

        $space = Space::Where([['uuid', $space_uuid], ['is_deleted', 0], ['is_active', 1]])->first();
        if (!$space) {
            return $this->ReturnResponse(false, "Space not found", [], 404);
        }
        $order = Order::select('orders.*') //
            ->join('order_statuses', 'order_statuses.id', '=', 'orders.order_status_id')
            ->join('line_items', 'line_items.order_id', '=', 'orders.id')
            ->join('stocks', 'stocks.id', '=', 'line_items.stock_id')
            ->with(['orderStatus', 'lineItems.stock.item'])
            ->where([['orders.is_deleted', 0], ['orders.is_active', 1]])
            ->where([
                ['stocks.space_id', $space->id],
                ['orders.uuid', $uuid],
                //['order_status_id', OrderStatus::where('name','!=','CART')->first()->id]
            ])->first();



        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        $validator = Validator::make(
            $request->all(),
            [
                'space_id'                 => 'required|exists:spaces,id',
                'line_items'               => 'required|array',
                'line_items.*.stock_id'    => 'required|exists:stocks,id',
                'line_items.*.quantity'    => 'required|min:1',
            ]
        );
        //si erreur
        if ($validator->fails()) {
            return $this->ReturnResponse(false, $validator->errors(), [], 400);
        }


        $stock_id_list   = [];
        $list_line_items = [];

        foreach ($request->line_items as $key => $sline) {
            $stock_id_list[$request->line_items[$key]["stock_id"]] = $sline["quantity"]; // si deux element identique faire la somme
        }

        try {
            DB::beginTransaction();

            $stocks = Stock::where('space_id', $request->space_id)->whereIn('id', array_keys($stock_id_list))->get();

            if (count($stocks) != count($stock_id_list)) {
                return $this->ReturnResponse(false, "Certains articles n'appartiennent pas à cet espace", [], 404);
            }

            $order->lineItems()->delete();

            $amount_due = 0;


            foreach ($stocks as $key => $stock) {
                $line = [
                    "stock_id"      => $stock->id,
                    "order_id"      => $order->id,
                    "quantity"      => $stock_id_list[$stock->id],
                    "created_by"    => $request->user()->id,
                    "updated_by"    => $request->user()->id,
                ];

                $amount_due += $stock->price_after_sale * $stock_id_list[$stock->id];
                $list_line_items[] = $line;
            }

            LineItem::insert($list_line_items);

            $order->update([
                "amount_due"        => $amount_due,
                "remaining_amount"  => $amount_due - $order->rebate,
            ]);
            $order = Order::where('id', $order->id)->with(['orderStatus', 'lineItems'])->first();

            DB::commit();
            return $this->ReturnResponse(true, "La commande a été mise à jour", [$order->with('lineItems')]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [], 500);
        }
    }



    public function pay(Request $request, string $uuid): JsonResponse
    {
        $data = [
            "id" => 1,
            "uuid" => Str::uuid(),
            "amount_paid" => 500,
        ];
        return $this->ReturnResponse(true, "Order Payed Successfully", $data);
    }

    public function deleteOrder(Request $request, string $uuid): JsonResponse
    {
        $order = Order::where([
            ['uuid', $uuid],
            ['is_deleted', 0],
            ['order_status_id', OrderStatus::where('name', 'CONFIRMED')->first()->id]
        ])->first();

        if (!$order) {
            return $this->ReturnResponse(false, "Order not found", [], 404);
        }

        try {
            DB::beginTransaction();
            $order->order_status_id = OrderStatus::where('name', 'CANCELED')->first()->id;
            $order->updated_at  = now();
            $order->updated_by  = $request->user()->id;

            $order->save();
            DB::commit();
            return $this->ReturnResponse(true, "Order Updated", [$order]);
        } catch (Exception $e) {
            DB::rollBack();
            // Log Error
            $message =  "Impossibme de mettre a jour la commande :  " . $e->getMessage() . " - line: " . $e->getLine();
            //Log::channel('customlog')->error($message);
            return $this->ReturnResponse(false, $message, [$order], 500);
        }

        return $this->ReturnResponse(true, "Order Deleted Successfully");
    }

    public function getBasketByUser(Request $request)
    {
        $orders = Order::where([
            ['created_by', $request->user()->id],
            ['order_status_id', OrderStatus::where('name', 'CART')->first()->id]
        ])->with(['lineItems'])->first();
        $basket = [];
        if ($orders) {
            foreach ($orders->lineItems as $line) {
                $stock = Stock::where('id', $line->stock_id)->with('item')->first();
                if (!empty($stock)) {
                    $p = array(
                        "count" => $line->quantity,
                        "description" => $stock->item->description,
                        "image" => $stock->item->img_1,
                        "is_prom" => $stock->item->is_on_promotion,
                        "name" => $stock->item->name,
                        "price" => $stock->price,
                        "quantity" => $stock->quantity,
                        "reduction" => $stock->off_by_number,
                        "reductionType" => 1,
                    );
                    array_push($basket, $p);
                }
            }
        }
        return $basket;
    }
}
